/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.core.io.buffer;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.LockSupport;
import java.util.function.Consumer;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferFactory;
import org.springframework.lang.Nullable;

final class OutputStreamPublisher
implements Publisher<DataBuffer> {
    private final Consumer<OutputStream> outputStreamConsumer;
    private final DataBufferFactory bufferFactory;
    private final Executor executor;
    private final int chunkSize;

    OutputStreamPublisher(Consumer<OutputStream> outputStreamConsumer, DataBufferFactory bufferFactory, Executor executor, int chunkSize) {
        this.outputStreamConsumer = outputStreamConsumer;
        this.bufferFactory = bufferFactory;
        this.executor = executor;
        this.chunkSize = chunkSize;
    }

    public void subscribe(Subscriber<? super DataBuffer> subscriber) {
        Objects.requireNonNull(subscriber, "Subscriber must not be null");
        OutputStreamSubscription subscription = new OutputStreamSubscription(subscriber, this.outputStreamConsumer, this.bufferFactory, this.chunkSize);
        subscriber.onSubscribe((Subscription)subscription);
        this.executor.execute(subscription::invokeHandler);
    }

    private static final class OutputStreamSubscription
    extends OutputStream
    implements Subscription {
        private static final Object READY = new Object();
        private final Subscriber<? super DataBuffer> actual;
        private final Consumer<OutputStream> outputStreamHandler;
        private final DataBufferFactory bufferFactory;
        private final int chunkSize;
        private final AtomicLong requested = new AtomicLong();
        private final AtomicReference<Object> parkedThread = new AtomicReference();
        @Nullable
        private volatile Throwable error;
        private long produced;

        OutputStreamSubscription(Subscriber<? super DataBuffer> actual, Consumer<OutputStream> outputStreamConsumer, DataBufferFactory bufferFactory, int chunkSize) {
            this.actual = actual;
            this.outputStreamHandler = outputStreamConsumer;
            this.bufferFactory = bufferFactory;
            this.chunkSize = chunkSize;
        }

        @Override
        public void write(int b) throws IOException {
            this.checkDemandAndAwaitIfNeeded();
            DataBuffer next = this.bufferFactory.allocateBuffer(1);
            next.write((byte)b);
            this.actual.onNext((Object)next);
            ++this.produced;
        }

        @Override
        public void write(byte[] b) throws IOException {
            this.write(b, 0, b.length);
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            this.checkDemandAndAwaitIfNeeded();
            DataBuffer next = this.bufferFactory.allocateBuffer(len);
            next.write(b, off, len);
            this.actual.onNext((Object)next);
            ++this.produced;
        }

        private void checkDemandAndAwaitIfNeeded() throws IOException {
            long r = this.requested.get();
            if (OutputStreamSubscription.isTerminated(r) || OutputStreamSubscription.isCancelled(r)) {
                throw new IOException("Subscription has been terminated");
            }
            long p = this.produced;
            if (p == r) {
                if (p > 0L) {
                    r = this.tryProduce(p);
                    this.produced = 0L;
                }
                while (true) {
                    if (OutputStreamSubscription.isTerminated(r) || OutputStreamSubscription.isCancelled(r)) {
                        throw new IOException("Subscription has been terminated");
                    }
                    if (r != 0L) {
                        return;
                    }
                    this.await();
                    r = this.requested.get();
                }
            }
        }

        private void invokeHandler() {
            Throwable error;
            try (BufferedOutputStream outputStream = new BufferedOutputStream(this, this.chunkSize);){
                this.outputStreamHandler.accept(outputStream);
            }
            catch (Exception ex) {
                Throwable error2;
                long previousState = this.tryTerminate();
                if (OutputStreamSubscription.isCancelled(previousState)) {
                    return;
                }
                if (OutputStreamSubscription.isTerminated(previousState) && (error2 = this.error) != null) {
                    this.actual.onError(error2);
                    return;
                }
                this.actual.onError((Throwable)ex);
                return;
            }
            long previousState = this.tryTerminate();
            if (OutputStreamSubscription.isCancelled(previousState)) {
                return;
            }
            if (OutputStreamSubscription.isTerminated(previousState) && (error = this.error) != null) {
                this.actual.onError(error);
                return;
            }
            this.actual.onComplete();
        }

        public void request(long n) {
            if (n <= 0L) {
                this.error = new IllegalArgumentException("request should be a positive number");
                long previousState = this.tryTerminate();
                if (OutputStreamSubscription.isTerminated(previousState) || OutputStreamSubscription.isCancelled(previousState)) {
                    return;
                }
                if (previousState > 0L) {
                    return;
                }
                this.resume();
                return;
            }
            if (this.addCap(n) == 0L) {
                this.resume();
            }
        }

        public void cancel() {
            long previousState = this.tryCancel();
            if (OutputStreamSubscription.isCancelled(previousState) || previousState > 0L) {
                return;
            }
            this.resume();
        }

        private void await() {
            Object current;
            Thread toUnpark = Thread.currentThread();
            while ((current = this.parkedThread.get()) != READY) {
                if (current != null && current != toUnpark) {
                    throw new IllegalStateException("Only one (Virtual)Thread can await!");
                }
                if (!this.parkedThread.compareAndSet(null, toUnpark)) continue;
                LockSupport.park();
            }
            this.parkedThread.lazySet(null);
        }

        private void resume() {
            Object old;
            if (this.parkedThread.get() != READY && (old = this.parkedThread.getAndSet(READY)) != READY) {
                LockSupport.unpark((Thread)old);
            }
        }

        private long tryCancel() {
            long r;
            do {
                if (!OutputStreamSubscription.isCancelled(r = this.requested.get())) continue;
                return r;
            } while (!this.requested.compareAndSet(r, Long.MIN_VALUE));
            return r;
        }

        private long tryTerminate() {
            long r;
            do {
                if (!OutputStreamSubscription.isCancelled(r = this.requested.get()) && !OutputStreamSubscription.isTerminated(r)) continue;
                return r;
            } while (!this.requested.compareAndSet(r, -1L));
            return r;
        }

        private long tryProduce(long n) {
            long update;
            long current;
            do {
                if (OutputStreamSubscription.isTerminated(current = this.requested.get()) || OutputStreamSubscription.isCancelled(current)) {
                    return current;
                }
                if (current == Long.MAX_VALUE) {
                    return Long.MAX_VALUE;
                }
                update = current - n;
                if (update >= 0L) continue;
                update = 0L;
            } while (!this.requested.compareAndSet(current, update));
            return update;
        }

        private long addCap(long n) {
            long u;
            long r;
            do {
                if (!OutputStreamSubscription.isTerminated(r = this.requested.get()) && !OutputStreamSubscription.isCancelled(r) && r != Long.MAX_VALUE) continue;
                return r;
            } while (!this.requested.compareAndSet(r, u = OutputStreamSubscription.addCap(r, n)));
            return r;
        }

        private static boolean isTerminated(long state) {
            return state == -1L;
        }

        private static boolean isCancelled(long state) {
            return state == Long.MIN_VALUE;
        }

        private static long addCap(long a, long b) {
            long res = a + b;
            if (res < 0L) {
                return Long.MAX_VALUE;
            }
            return res;
        }
    }
}

