/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.google.common.primitives.Ints;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.PartitionPosition;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.LeveledGenerations;
import org.apache.cassandra.db.compaction.SizeTieredCompactionStrategy;
import org.apache.cassandra.db.compaction.SizeTieredCompactionStrategyOptions;
import org.apache.cassandra.dht.Bounds;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.service.StorageService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeveledManifest {
    private static final Logger logger = LoggerFactory.getLogger(LeveledManifest.class);
    private static final int MAX_COMPACTING_L0 = 32;
    private static final int NO_COMPACTION_LIMIT = 25;
    private final ColumnFamilyStore cfs;
    private final LeveledGenerations generations;
    private final SSTableReader[] lastCompactedSSTables;
    private final long maxSSTableSizeInBytes;
    private final SizeTieredCompactionStrategyOptions options;
    private final int[] compactionCounter;
    private final int levelFanoutSize;

    LeveledManifest(ColumnFamilyStore cfs, int maxSSTableSizeInMB, int fanoutSize, SizeTieredCompactionStrategyOptions options) {
        this.cfs = cfs;
        this.maxSSTableSizeInBytes = (long)maxSSTableSizeInMB * 1024L * 1024L;
        this.options = options;
        this.levelFanoutSize = fanoutSize;
        this.lastCompactedSSTables = new SSTableReader[9];
        this.generations = new LeveledGenerations();
        this.compactionCounter = new int[9];
    }

    public static LeveledManifest create(ColumnFamilyStore cfs, int maxSSTableSize, int fanoutSize, List<SSTableReader> sstables) {
        return LeveledManifest.create(cfs, maxSSTableSize, fanoutSize, sstables, new SizeTieredCompactionStrategyOptions());
    }

    public static LeveledManifest create(ColumnFamilyStore cfs, int maxSSTableSize, int fanoutSize, Iterable<SSTableReader> sstables, SizeTieredCompactionStrategyOptions options) {
        LeveledManifest manifest = new LeveledManifest(cfs, maxSSTableSize, fanoutSize, options);
        manifest.addSSTables(sstables);
        manifest.calculateLastCompactedKeys();
        return manifest;
    }

    void calculateLastCompactedKeys() {
        for (int i = 0; i < this.generations.levelCount() - 1; ++i) {
            Set<SSTableReader> level = this.generations.get(i + 1);
            if (level.isEmpty()) continue;
            SSTableReader sstableWithMaxModificationTime = null;
            long maxModificationTime = Long.MIN_VALUE;
            for (SSTableReader ssTableReader : level) {
                long modificationTime = ssTableReader.getCreationTimeFor(Component.DATA);
                if (modificationTime < maxModificationTime) continue;
                sstableWithMaxModificationTime = ssTableReader;
                maxModificationTime = modificationTime;
            }
            this.lastCompactedSSTables[i] = sstableWithMaxModificationTime;
        }
    }

    public synchronized void addSSTables(Iterable<SSTableReader> readers) {
        this.generations.addAll(readers);
    }

    public synchronized void replace(Collection<SSTableReader> removed, Collection<SSTableReader> added) {
        assert (!removed.isEmpty());
        if (logger.isTraceEnabled()) {
            this.generations.logDistribution();
            logger.trace("Replacing [{}]", (Object)this.toString(removed));
        }
        int minLevel = this.generations.remove(removed);
        if (added.isEmpty()) {
            return;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Adding [{}]", (Object)this.toString(added));
        }
        this.generations.addAll(added);
        this.lastCompactedSSTables[minLevel] = (SSTableReader)SSTableReader.sstableOrdering.max(added);
    }

    private String toString(Collection<SSTableReader> sstables) {
        StringBuilder builder = new StringBuilder();
        for (SSTableReader sstable : sstables) {
            builder.append(sstable.descriptor.cfname).append('-').append(sstable.descriptor.generation).append("(L").append(sstable.getSSTableLevel()).append("), ");
        }
        return builder.toString();
    }

    public long maxBytesForLevel(int level, long maxSSTableSizeInBytes) {
        return LeveledManifest.maxBytesForLevel(level, this.levelFanoutSize, maxSSTableSizeInBytes);
    }

    public static long maxBytesForLevel(int level, int levelFanoutSize, long maxSSTableSizeInBytes) {
        if (level == 0) {
            return 4L * maxSSTableSizeInBytes;
        }
        double bytes = Math.pow(levelFanoutSize, level) * (double)maxSSTableSizeInBytes;
        if (bytes > 9.223372036854776E18) {
            throw new RuntimeException("At most 9223372036854775807 bytes may be in a compaction level; your maxSSTableSize must be absurdly high to compute " + bytes);
        }
        return (long)bytes;
    }

    public synchronized CompactionCandidate getCompactionCandidates() {
        if (StorageService.instance.isBootstrapMode()) {
            List<SSTableReader> mostInteresting = this.getSSTablesForSTCS(this.generations.get(0));
            if (!mostInteresting.isEmpty()) {
                logger.info("Bootstrapping - doing STCS in L0");
                return new CompactionCandidate(mostInteresting, 0, Long.MAX_VALUE);
            }
            return null;
        }
        CompactionCandidate l0Compaction = this.getSTCSInL0CompactionCandidate();
        for (int i = this.generations.levelCount() - 1; i > 0; --i) {
            Set<SSTableReader> sstables = this.generations.get(i);
            if (sstables.isEmpty()) continue;
            HashSet sstablesInLevel = Sets.newHashSet(sstables);
            Sets.SetView remaining = Sets.difference((Set)sstablesInLevel, this.cfs.getTracker().getCompacting());
            long remainingBytesForLevel = SSTableReader.getTotalBytes((Iterable<SSTableReader>)remaining);
            long maxBytesForLevel = this.maxBytesForLevel(i, this.maxSSTableSizeInBytes);
            double score = (double)remainingBytesForLevel / (double)maxBytesForLevel;
            logger.trace("Compaction score for level {} is {}", (Object)i, (Object)score);
            if (!(score > 1.001)) continue;
            if (i == this.generations.levelCount() - 1) {
                logger.warn("L" + i + " (maximum supported level) has " + remainingBytesForLevel + " bytes while its maximum size is supposed to be " + maxBytesForLevel + " bytes");
                continue;
            }
            if (l0Compaction != null) {
                return l0Compaction;
            }
            Collection<SSTableReader> candidates = this.getCandidatesFor(i);
            if (!candidates.isEmpty()) {
                int nextLevel = this.getNextLevel(candidates);
                candidates = this.getOverlappingStarvedSSTables(nextLevel, candidates);
                if (logger.isTraceEnabled()) {
                    logger.trace("Compaction candidates for L{} are {}", (Object)i, (Object)this.toString(candidates));
                }
                return new CompactionCandidate(candidates, nextLevel, this.maxSSTableSizeInBytes);
            }
            logger.trace("No compaction candidates for L{}", (Object)i);
        }
        if (this.generations.get(0).isEmpty()) {
            return null;
        }
        Collection<SSTableReader> candidates = this.getCandidatesFor(0);
        if (candidates.isEmpty()) {
            return l0Compaction;
        }
        return new CompactionCandidate(candidates, this.getNextLevel(candidates), this.maxSSTableSizeInBytes);
    }

    private CompactionCandidate getSTCSInL0CompactionCandidate() {
        List<SSTableReader> mostInteresting;
        if (!DatabaseDescriptor.getDisableSTCSInL0() && this.generations.get(0).size() > 32 && !(mostInteresting = this.getSSTablesForSTCS(this.generations.get(0))).isEmpty()) {
            logger.debug("L0 is too far behind, performing size-tiering there first");
            return new CompactionCandidate(mostInteresting, 0, Long.MAX_VALUE);
        }
        return null;
    }

    private List<SSTableReader> getSSTablesForSTCS(Collection<SSTableReader> sstables) {
        Iterable<SSTableReader> candidates = this.cfs.getTracker().getUncompacting(sstables);
        List pairs = SizeTieredCompactionStrategy.createSSTableAndLengthPairs(AbstractCompactionStrategy.filterSuspectSSTables(candidates));
        List<List<SSTableReader>> buckets = SizeTieredCompactionStrategy.getBuckets(pairs, this.options.bucketHigh, this.options.bucketLow, this.options.minSSTableSize);
        return SizeTieredCompactionStrategy.mostInterestingBucket(buckets, this.cfs.getMinimumCompactionThreshold(), this.cfs.getMaximumCompactionThreshold());
    }

    private Collection<SSTableReader> getOverlappingStarvedSSTables(int targetLevel, Collection<SSTableReader> candidates) {
        HashSet<SSTableReader> withStarvedCandidate = new HashSet<SSTableReader>(candidates);
        int i = this.generations.levelCount() - 1;
        while (i > 0) {
            int n = i--;
            this.compactionCounter[n] = this.compactionCounter[n] + 1;
        }
        this.compactionCounter[targetLevel] = 0;
        if (logger.isTraceEnabled()) {
            for (int j = 0; j < this.compactionCounter.length; ++j) {
                logger.trace("CompactionCounter: {}: {}", (Object)j, (Object)this.compactionCounter[j]);
            }
        }
        for (i = this.generations.levelCount() - 1; i > 0; --i) {
            if (this.getLevelSize(i) <= 0) continue;
            if (this.compactionCounter[i] > 25) {
                DecoratedKey max = null;
                Object min = null;
                for (SSTableReader candidate : candidates) {
                    if (min == null || candidate.first.compareTo((PartitionPosition)min) < 0) {
                        min = candidate.first;
                    }
                    if (max != null && candidate.last.compareTo(max) <= 0) continue;
                    max = candidate.last;
                }
                if (min == null || max == null || min.equals(max)) {
                    return candidates;
                }
                Set<SSTableReader> compacting = this.cfs.getTracker().getCompacting();
                Range<DecoratedKey> boundaries = new Range<DecoratedKey>((DecoratedKey)min, max);
                for (SSTableReader sstable : this.generations.get(i)) {
                    Range<DecoratedKey> r = new Range<DecoratedKey>(sstable.first, sstable.last);
                    if (!boundaries.contains((DecoratedKey)((Object)r)) || compacting.contains(sstable)) continue;
                    logger.info("Adding high-level (L{}) {} to candidates", (Object)sstable.getSSTableLevel(), (Object)sstable);
                    withStarvedCandidate.add(sstable);
                    return withStarvedCandidate;
                }
            }
            return candidates;
        }
        return candidates;
    }

    public synchronized int getLevelSize(int i) {
        return this.generations.get(i).size();
    }

    public synchronized int[] getAllLevelSize() {
        return this.generations.getAllLevelSize();
    }

    @VisibleForTesting
    public synchronized int remove(SSTableReader reader) {
        int level = reader.getSSTableLevel();
        assert (level >= 0) : reader + " not present in manifest: " + level;
        this.generations.remove(Collections.singleton(reader));
        return level;
    }

    public synchronized Set<SSTableReader> getSSTables() {
        return this.generations.allSSTables();
    }

    private static Set<SSTableReader> overlapping(Collection<SSTableReader> candidates, Iterable<SSTableReader> others) {
        assert (!candidates.isEmpty());
        Iterator<SSTableReader> iter = candidates.iterator();
        SSTableReader sstable = iter.next();
        Token first = sstable.first.getToken();
        Token last = sstable.last.getToken();
        while (iter.hasNext()) {
            sstable = iter.next();
            first = first.compareTo(sstable.first.getToken()) <= 0 ? first : sstable.first.getToken();
            last = last.compareTo(sstable.last.getToken()) >= 0 ? last : sstable.last.getToken();
        }
        return LeveledManifest.overlapping(first, last, others);
    }

    private static Set<SSTableReader> overlappingWithBounds(SSTableReader sstable, Map<SSTableReader, Bounds<Token>> others) {
        return LeveledManifest.overlappingWithBounds(sstable.first.getToken(), sstable.last.getToken(), others);
    }

    @VisibleForTesting
    static Set<SSTableReader> overlapping(Token start, Token end, Iterable<SSTableReader> sstables) {
        return LeveledManifest.overlappingWithBounds(start, end, LeveledManifest.genBounds(sstables));
    }

    private static Set<SSTableReader> overlappingWithBounds(Token start, Token end, Map<SSTableReader, Bounds<Token>> sstables) {
        assert (start.compareTo(end) <= 0);
        HashSet<SSTableReader> overlapped = new HashSet<SSTableReader>();
        Bounds<Token> promotedBounds = new Bounds<Token>(start, end);
        for (Map.Entry<SSTableReader, Bounds<Token>> pair : sstables.entrySet()) {
            if (!pair.getValue().intersects(promotedBounds)) continue;
            overlapped.add(pair.getKey());
        }
        return overlapped;
    }

    private static Map<SSTableReader, Bounds<Token>> genBounds(Iterable<SSTableReader> ssTableReaders) {
        HashMap<SSTableReader, Bounds<Token>> boundsMap = new HashMap<SSTableReader, Bounds<Token>>();
        for (SSTableReader sstable : ssTableReaders) {
            boundsMap.put(sstable, new Bounds<Token>(sstable.first.getToken(), sstable.last.getToken()));
        }
        return boundsMap;
    }

    private Collection<SSTableReader> getCandidatesFor(int level) {
        assert (!this.generations.get(level).isEmpty());
        logger.trace("Choosing candidates for L{}", (Object)level);
        Set<SSTableReader> compacting = this.cfs.getTracker().getCompacting();
        if (level == 0) {
            Set<SSTableReader> compactingL0 = this.getCompactingL0();
            DecoratedKey lastCompactingKey = null;
            DecoratedKey firstCompactingKey = null;
            for (SSTableReader candidate : compactingL0) {
                if (firstCompactingKey == null || candidate.first.compareTo(firstCompactingKey) < 0) {
                    firstCompactingKey = candidate.first;
                }
                if (lastCompactingKey != null && candidate.last.compareTo(lastCompactingKey) <= 0) continue;
                lastCompactingKey = candidate.last;
            }
            Object candidates = new HashSet();
            Map<SSTableReader, Bounds<Token>> remaining = LeveledManifest.genBounds(Iterables.filter(this.generations.get(0), (Predicate)Predicates.not(SSTableReader::isMarkedSuspect)));
            for (SSTableReader sstable : this.ageSortedSSTables(remaining.keySet())) {
                Sets.SetView overlappedL0;
                if (candidates.contains(sstable) || !Sets.intersection((Set)(overlappedL0 = Sets.union(Collections.singleton(sstable), LeveledManifest.overlappingWithBounds(sstable, remaining))), compactingL0).isEmpty()) continue;
                for (SSTableReader newCandidate : overlappedL0) {
                    if (firstCompactingKey == null || lastCompactingKey == null || LeveledManifest.overlapping(firstCompactingKey.getToken(), lastCompactingKey.getToken(), Collections.singleton(newCandidate)).size() == 0) {
                        candidates.add(newCandidate);
                    }
                    remaining.remove(newCandidate);
                }
                if (candidates.size() <= this.cfs.getMaximumCompactionThreshold()) continue;
                candidates = new HashSet<SSTableReader>(this.ageSortedSSTables((Collection<SSTableReader>)candidates).subList(0, this.cfs.getMaximumCompactionThreshold()));
                break;
            }
            if (SSTableReader.getTotalBytes(candidates) > this.maxSSTableSizeInBytes) {
                Set<SSTableReader> l1overlapping = LeveledManifest.overlapping(candidates, this.generations.get(1));
                if (Sets.intersection(l1overlapping, compacting).size() > 0) {
                    return Collections.emptyList();
                }
                if (!LeveledManifest.overlapping(candidates, compactingL0).isEmpty()) {
                    return Collections.emptyList();
                }
                candidates = Sets.union(candidates, l1overlapping);
            }
            if (candidates.size() < 2) {
                return Collections.emptyList();
            }
            return candidates;
        }
        Map<SSTableReader, Bounds<Token>> sstablesNextLevel = LeveledManifest.genBounds(this.generations.get(level + 1));
        Iterator<SSTableReader> levelIterator = this.generations.wrappingIterator(level, this.lastCompactedSSTables[level]);
        while (levelIterator.hasNext()) {
            SSTableReader sstable = levelIterator.next();
            Sets.SetView candidates = Sets.union(Collections.singleton(sstable), LeveledManifest.overlappingWithBounds(sstable, sstablesNextLevel));
            if (Iterables.any((Iterable)candidates, SSTableReader::isMarkedSuspect) || !Sets.intersection((Set)candidates, compacting).isEmpty()) continue;
            return candidates;
        }
        return Collections.emptyList();
    }

    private Set<SSTableReader> getCompactingL0() {
        HashSet<SSTableReader> sstables = new HashSet<SSTableReader>();
        HashSet<SSTableReader> levelSSTables = new HashSet<SSTableReader>(this.generations.get(0));
        for (SSTableReader sstable : this.cfs.getTracker().getCompacting()) {
            if (!levelSSTables.contains(sstable)) continue;
            sstables.add(sstable);
        }
        return sstables;
    }

    @VisibleForTesting
    List<SSTableReader> ageSortedSSTables(Collection<SSTableReader> candidates) {
        return ImmutableList.sortedCopyOf(SSTableReader.maxTimestampAscending, candidates);
    }

    public synchronized Set<SSTableReader>[] getSStablesPerLevelSnapshot() {
        return this.generations.snapshot();
    }

    public String toString() {
        return "Manifest@" + this.hashCode();
    }

    public synchronized int getLevelCount() {
        for (int i = this.generations.levelCount() - 1; i >= 0; --i) {
            if (this.generations.get(i).size() <= 0) continue;
            return i;
        }
        return 0;
    }

    public synchronized int getEstimatedTasks() {
        long tasks = 0L;
        long[] estimated = new long[this.generations.levelCount()];
        for (int i = this.generations.levelCount() - 1; i >= 0; --i) {
            Set<SSTableReader> sstables = this.generations.get(i);
            estimated[i] = (long)Math.ceil((double)Math.max(0L, SSTableReader.getTotalBytes(sstables) - (long)((double)this.maxBytesForLevel(i, this.maxSSTableSizeInBytes) * 1.001)) / (double)this.maxSSTableSizeInBytes);
            tasks += estimated[i];
        }
        if (!DatabaseDescriptor.getDisableSTCSInL0() && this.generations.get(0).size() > this.cfs.getMaximumCompactionThreshold()) {
            int l0compactions = this.generations.get(0).size() / this.cfs.getMaximumCompactionThreshold();
            tasks += (long)l0compactions;
            estimated[0] = estimated[0] + (long)l0compactions;
        }
        logger.trace("Estimating {} compactions to do for {}.{}", new Object[]{Arrays.toString(estimated), this.cfs.keyspace.getName(), this.cfs.name});
        return Ints.checkedCast((long)tasks);
    }

    public int getNextLevel(Collection<SSTableReader> sstables) {
        int newLevel;
        int maximumLevel = Integer.MIN_VALUE;
        int minimumLevel = Integer.MAX_VALUE;
        for (SSTableReader sstable : sstables) {
            maximumLevel = Math.max(sstable.getSSTableLevel(), maximumLevel);
            minimumLevel = Math.min(sstable.getSSTableLevel(), minimumLevel);
        }
        if (minimumLevel == 0 && minimumLevel == maximumLevel && SSTableReader.getTotalBytes(sstables) < this.maxSSTableSizeInBytes) {
            newLevel = 0;
        } else {
            int n = newLevel = minimumLevel == maximumLevel ? maximumLevel + 1 : maximumLevel;
            assert (newLevel > 0);
        }
        return newLevel;
    }

    synchronized Set<SSTableReader> getLevel(int level) {
        return ImmutableSet.copyOf(this.generations.get(level));
    }

    synchronized List<SSTableReader> getLevelSorted(int level, Comparator<SSTableReader> comparator) {
        return ImmutableList.sortedCopyOf(comparator, this.generations.get(level));
    }

    synchronized void newLevel(SSTableReader sstable, int oldLevel) {
        this.generations.newLevel(sstable, oldLevel);
        this.lastCompactedSSTables[oldLevel] = sstable;
    }

    public static class CompactionCandidate {
        public final Collection<SSTableReader> sstables;
        public final int level;
        public final long maxSSTableBytes;

        public CompactionCandidate(Collection<SSTableReader> sstables, int level, long maxSSTableBytes) {
            this.sstables = sstables;
            this.level = level;
            this.maxSSTableBytes = maxSSTableBytes;
        }
    }
}

